<?php
namespace Wizhou\Magasin;

use Wizhou\Magasin\Magasin;
use Wizhou\Magasin\Currency;
use Wizhou\Magasin\Shipping;
use Kirby\Toolkit\Collection;
use Kirby\Exception\Exception;
use Kirby\Toolkit\A;

class CartItem
{
  protected $id;
  protected $priceBeforeTax;
  protected $tax;
  protected $priceIncludingTax;
  protected $quantity;
  protected $stock;
  protected $weight;
  protected $width;
  protected $height;
  protected $productDepth;
  protected $sumBeforeTax;
  protected $sumIncludingTax;
  protected $taxAmount;
  protected $amount;
  protected $sumAmount;
  protected $shippingAdditionalCosts;


  /**
   * [__construct Fill the Cart item with data from the product page]
   * @param Kriby/Cms/Page | String $page [Product page]
   * @param Int $quantity [Quantity of the item]
   */

  public function __construct($page, Int $quantity) {
    try {

      if (is_string($page)) {
        try {
          $page = kirby()->site()->page($page);
        } catch (\Exception $e) {
          throw new \Exception("Page doesn't exist", 1);
        }
      }

      if ($page->intendedTemplate()->name() !== 'product') {
        throw new \Exception("Page is not a product page", 1);
      }

      /* Add Id to the item */
      if ($page->id()) {
        $this->id = $page->id();
      } else {
        throw new \Exception("Cart Item has no id", 1);
      }

      if ($page->price_before_tax()->isNotEmpty()) {
        $this->priceBeforeTax = $page->price_before_tax()->toFloat();
      } else {
        throw new \Exception("Product doesn't have a price without tax", 1);
      }

      if ($page->tax()->isNotEmpty()) {
        $this->tax = $page->tax()->toInt();
      } else {
        throw new \Exception("Product doesn't have a tax", 1);
      }

      if ($page->price_including_tax()->isNotEmpty()) {
        $this->priceIncludingTax = $page->price_including_tax()->toFloat();
      } else {
        throw new \Exception("Product doesn't have a price including tax", 1);
      }

      if (isset($quantity)) {
        $this->quantity = $quantity;
      } else {
        throw new \Exception("Cart item need to have a quantity", 1);
      }

      if ($page->stock()->isNotEmpty() &&
      $page->stock()->toInt() > 0) {
        $stock = $page->stock()->toInt();
        $this->stock = $stock;
      } else {
        throw new \Exception("The product has no stock", 1);
      }

      if ($page->weight()->isNotEmpty()) {
        $this->weight = $page->weight()->toFloat();
      }

      if ($page->width()->isNotEmpty()) {
        $this->width = $page->width()->toFloat();
      }

      if ($page->height()->isNotEmpty()) {
        $this->height = $page->height()->toFloat();
      }

      if ($page->product_depth()->isNotEmpty()) {
        $this->productDepth = $page->product_depth()->toFloat();
      }

      if ($this->hasEnoughStock() === false) {
        throw new \Exception("There is not enough stock for the desired quantity", 1);
      }

      if ($page->shipping_additional_costs()->isNotEmpty()) {
        $this->shippingAdditionalCosts = $page->shipping_additional_costs()->toFloat();
      }

      /* Get the price and taxes based on the data */
      $this->makePriceCalculation();

    } catch (\Exception $e) {
        throw new \Exception($e);
    }
  }

  public function __destruct() {
  }

  /**
   * [hasEnoughStock Check if the product have enough stock for the desired quantity]
   * @return boolean
   */

  public function hasEnoughStock() {
    $stock = $this->stock();
    $quantity = $this->quantity();

    if ($stock >= $quantity) {
      return true;
    } else {
      return false;
    }
  }

  /**
   * [checkTheStock Safe guard for the stock with error]
   * @return boolean [True or Error]
   */

  public function checkTheStock() {
    if ($this->hasEnoughStock()) {
      return true;
    } else {
      throw new \Exception("There is not enough stock for the desired quantity", 1);
    }
  }

  /**
   * [update Chage data from the Cart item ]
   * @param  array  $params [Fields to change with value]
   */

  public function update(array $params) {
    try {
      foreach ($params as $key => $value) {
        switch ($key) {
          case 'id':
            if (is_string($value)) {
              if (kirby()->site()->page($value)) {
                $this->id = $value;
              } else {
                throw new \Exception("This page id doesn't exist", 1);
              }
            } else {
              throw new \Exception("Id must be a String", 1);
            }
          break;

          case 'priceBeforeTax':
            if (is_float($value)) {
              $this->priceBeforeTax = $value;

              $priceIncludingTax = Currency::getPriceIncludingTax($value, $this->tax());
              $this->priceIncludingTax = $priceIncludingTax;
              $this->makePriceCalculation();
            } else {
              throw new \Exception("Price must be a Float", 1);
            }
            break;

          case 'tax':
            if (is_int($value)) {
              $this->tax = $value;
              $priceBeforeTax = $this->priceBeforeTax();
              $priceIncludingTax =
              Currency::getPriceIncludingTax($priceBeforeTax, $value);
              $this->priceIncludingTax = $priceIncludingTax;
              $this->makePriceCalculation();
            } else {
              throw new \Error("Tax must be an Integrer", 1);
            }
            break;

          case 'priceIncludingTax':
            if (is_float($value)) {
              $this->priceIncludingTax = $value;
              $priceBeforeTax = Currency::getPriceBeforeTax($value, $this->tax());
              $this->priceBeforeTax = $priceBeforeTax;
              $this->makePriceCalculation();
            } else {
              throw new \Exception("Price must be a Float", 1);
            }
            break;

          case 'quantity':
            if (is_int($value)) {
              $this->quantity = $value;
              $this->checkTheStock();
              $this->makePriceCalculation();
            } else {
              throw new \Exception("Quantity must be an Integrer", 1);
            }
            break;

          case 'stock':
            if (is_int($value)) {
              $this->stock = $value;
              $this->checkTheStock();
            } else {
              throw new \Exception("Stock must be an Integrer", 1);
            }
            break;

          default:
            throw new \Exception("The value being update doesn't exist");
            break;
        }
      }
    } catch (\Exception $e) {
      throw new \Exception($e);
    }
  }

  /**
   * [makePriceCalculation Make calculation for the price and amount of the item]
   * @return Object [$this]
   */

  private function makePriceCalculation() {
    $priceBeforeTax = $this->priceBeforeTax();
    $priceIncludingTax = $this->priceIncludingTax();
    $tax = $this->tax();
    $quantity = $this->quantity();

    $sumBeforeTax = $priceBeforeTax * $quantity;
    $sumIncludingTax = $priceIncludingTax * $quantity;

    $taxAmount = Currency::getTaxAmout($sumBeforeTax, $tax);

    $this->sumBeforeTax = $sumBeforeTax;
    $this->sumIncludingTax = $sumIncludingTax;
    $this->taxAmount = $taxAmount;
    $this->amount = currency()->toAmount($priceIncludingTax);
    $this->sumAmount = currency()->toAmount($sumIncludingTax);
    return $this;
  }

  public function id() {
    return $this->id;
  }

  public function priceBeforeTax() {
    return $this->priceBeforeTax;
  }

  public function tax() {
    return $this->tax;
  }

  public function priceIncludingTax() {
    return $this->priceIncludingTax;
  }

  public function quantity() {
    return $this->quantity;
  }

  public function stock() {
    return $this->stock;
  }

  public function weight() {
    return $this->weight;
  }

  public function width() {
    return $this->width;
  }

  public function height() {
    return $this->height;
  }

  public function productDepth() {
    return $this->productDepth;
  }

  public function sumBeforeTax() {
    return $this->sumBeforeTax;
  }

  public function sumIncludingTax() {
    return $this->sumIncludingTax;
  }

  public function taxAmount() {
    return $this->taxAmount;
  }

  public function amount() {
    return $this->amount;
  }

  public function sumAmount() {
    return $this->sumAmount;
  }

  public function shippingAdditionalCosts() {
    return $this->shippingAdditionalCosts;
  }

  public function toPage() {
    return kirby()->site()->page($this->id());
  }
}
