<?php
namespace Wizhou\Magasin;
use Wizhou\Magasin\Currency;
use Wizhou\Magasin\ProductPage;
use Wizhou\Magasin\CartItem;
use Kirby\Exception\Exception;
use Kirby\Toolkit\Collection;
use Kirby\Toolkit\A;

class Cart extends Collection
{
  protected $sessionName = 'ww.magasin.cart';

  /**
   * [__construct Feed the cart with the save from the session if there is one.]
   * @param array $data  [Array of items if the cart need to be prebuilt]
   */

  public function __construct(array $data = []) {
    $kirby = kirby();

    if (count($data) === 0
    && is_array($kirby->session()->get($this->sessionName))) {
      $data = $kirby->session()->get($this->sessionName);
    }

    $this->data($data);
    return $this;
  }

  /**
   * [addItem Add a product to the Cart]
   * @param [Kriby/Cms/Page | String] $page [a page object with the Product template or a string with it's ID]
   * @param integer $quantity  [The quantity of product to add]
   */

  public function addItem($page, int $quantity = 1) {

    /* Safe guard */
    if(is_object($page)) {
      $page = $page->id();
    }

    /* If there is no quantity, remove the item or do nothing */
    if ($quantity === 0) {
      try {
        $this->removeItem($page);
        return $this;
      } catch (\Exception $e) {
        throw new \Exception("There is no quantity", 1);
      }
    }

    /* If the item is already there, update the quantity */
    if ($this->get($page)) {
      $cartItem = $this->get($page);
      $previousQuantity = $cartItem->quantity();
      $quantity = $previousQuantity + $quantity;

      /* Reset the quantity to max stock if needed */
      $stock = $cartItem->stock();
      if ($quantity > $stock) $quantity = $stock;
    }

    /* Add the item with desired quantity */
    try {
      $cartItem = new CartItem($page, $quantity);
      $this->append($cartItem->id(), $cartItem);
    } catch (\Exception $e) {
      throw new \Exception($e, 1);
    }

    $this->saveCart();
    return $this;
  }

  /**
   * [removeItem Remove an item from the Cart]
   * @param [Kriby/Cms/Page | String] $page [a page object with the Product template or a string with it's ID]
   * @return [$this] [Return the Cart]
   */

  public function removeItem($page) {
    try {
      /* Safe guard */
      if (is_object($page)) {
        $id = $page->id();
      } else {
        $id = $page;
      }

      if ($this->has($id)) {
        $this->remove($id);
      } else {
        throw new \Exception("This item is not present in the cart", 1);
      }
    } catch (\Exception $e) {
      throw new \Exception($e, 1);
    }

    $this->saveCart();
    return $this;
  }

  /**
   * [updateItem Update an item of the Cart]
   * @param [Kriby/Cms/Page | String] $page [a page object with the Product template or a string with it's ID]
   * @param  array  $params [Field / Value array withj the changes]
   * @return [$this] [Return the Cart]
   */

  public function updateItem($page, array $params) {
    try {
      /* Safe guard */
      if (is_object($page)) {
        $id = $page->id();
      } else {
        $id = $page;
      }

      if ($this->has($id)) {
        $item = $this->get($id);
        $item->update($params);
      } else {
        throw new \Exception("The item you are trying to update doesn't exist in the cart", 1);

      }
    } catch (\Exception $e) {
      throw new \Exception($e, 1);
    }

    $this->saveCart();
    return $this;
  }

  /**
   * [reload Reconstruct the cart]
   * @return [$this] [Return the Cart]
   */

  public function reload () {
    return $this->__construct();
  }

  /**
   * [saveCart Save the cart into the session]
   * @return [$this] [Return the Cart]
   */

  public function saveCart () {
    kirby()->session()->set($this->sessionName, $this->toArray());
      return $this;
  }

  /**
   * [quantityOfItems Get the quantity of all items in the Cart]
   * @return [int] [The quantity of items]
   */

  public function quantityOfItems() {
    $quantity = 0;
    foreach ($this as $key => $cartItem) {
      $quantity += $cartItem->quantity();
    }
    return (int) $quantity;
  }

  /**
   * [total Get total price of the Cart]
   * @return [float] [Total price of the cart]
   */

  public function total() {
    $total = 0;
    foreach ($this as $cartItem) {
      $total += $cartItem->sumIncludingTax();
    }
    return (float) $total;
  }

  /**
   * [prettyTotal Prettify the price for human reading]
   * @return [string] [Price formated with locale settings]
   */

  public function prettyTotal() {
    return currency()->prettify($this->total());
  }

  /**
   * [totalAmount Return the total of the Cart as an amount (to basics)]
   * @return [int] [Amount of the cart]
   */

  public function totalAmount() {
    $total = 0;
    foreach ($this as $cartItem) {
      $total += $cartItem->sumAmount();
    }
    return $total;
  }
  /**
   * [taxTotal Get the total price of tax from every item of the Cart]
   * @return [Number] [Total taxes amount]
   */

  public function taxTotal() {
    $total = 0;
    foreach ($this as $cartItem) {
      $total += $cartItem->tax();
    }
    return $total;
  }

  public function totalWeight() {
    $total = 0;
    foreach ($this as $cartItem) {
      $total += $cartItem->weight() * $cartItem->quantity();
    }
    return $total;
  }

  public function isSafeForShipping () {
    $shipping = shipping();

    foreach ($shipping as $carrier) {
      foreach ($this as $cartItem) {
        $check = $carrier->checkProductDimensions($cartItem);
        if (!$check) return false;
      }
      if (!$carrier->checkWeight($this->totalWeight())) {
        return false;
      }

      return true;
    }
  }

  public function isNotSafeForShipping () {
    return ! $this->isSafeForShipping();
  }

  public function shippingOptions () {
    $shipping = shipping();

    if ($this->isSafeForShipping()) {
      return $shipping->getShippingOptions($this);
    }
  }

  public function shippingPrice () {
    $shipping = shipping();

    if ($this->isSafeForShipping()) {
      /* Get shipping price for one carrier */
      $shippingPrice = $shipping->getShippingPrice($this);

      if ($shippingPrice === false) {
        /* If there is no option, retrun false */
        return false;
      }

      $additionalPrice = $this->getAdditionalShippingPrice();
      $total = $shippingPrice + $additionalPrice;

      return $total;
    }
  }

  public function getAdditionalShippingPrice() {
    $additionalPrice = 0;
    foreach ($this as $cartItem) {
      $additionalCosts = $cartItem->shippingAdditionalCosts();
      $additionalPrice += $additionalCosts * $cartItem->quantity();
    }

    return $additionalPrice;
  }

  public function finalTotal () {
    $cartPrice = $this->total();
    $shippingPrice = $this->shippingPrice();

    // TODO: Add elements for Promo code calculation

    $total = $cartPrice + $shippingPrice;
    return $total;
  }

  public function prettyShipping () {
    if ($this->shippingPrice() === false) {
      return 'Pas de livraison possible';
    } else {
      return currency()->prettify($this->shippingPrice());
    }
  }

  public function prettyFinalTotal () {
    return currency()->prettify($this->finalTotal());
  }

  /**
   * [flushCart Remove all item from the Cart and erase the session]
   */

  public function flushCart ():void {
    foreach ($this as $key => $value) {
      $this->removeItem($value);
    }

    kirby()->session()->remove($this->sessionName);
  }

  /**
   * [toData Build a data array with every item of the cart for json and ajax]
   * @return [Array] [Return all value of every item in a Json safe way]
   */

  public function toData () {
    $data = [];
    foreach ($this as $key => $value) {
      $data[$key] = [
        'id' => $value->id(),
        'priceBeforeTax' => $value->priceBeforeTax(),
        'tax' => $value->tax(),
        'priceIncludingTax' => $value->priceIncludingTax(),
        'quantity' => $value->quantity(),
        'stock' => $value->stock(),
        'weight' => $value->weight(),
        'sumIncludingTax' => $value->sumIncludingTax(),
        'sumBeforeTax' => $value->sumBeforeTax(),
        'taxAmount' => $value->taxAmount(),
        'amount' => $value->amount(),
        'sumAmount' => $value->sumAmount()
      ];
    }
    return $data;
  }

  /**
   * [toJsonForVue Return the cart in a Json safe way]
   * @return [Array] [All cart fields and calculation in a Json safe way.]
   */

  public function toJsonForVue () {
    return [
      'cart' => $this->toData(),
      'quantity_of_items' => $this->quantityOfItems(),
      'total' => $this->total(),
      'tax_total' => $this->taxTotal(),
      'shipping_total' => $this->shippingPrice(),
      'final_total' => $this->finalTotal()
    ];
  }
}
